/*
 * $Id$
 * 
 * created by Werum Software & Systems AG on 13.06.2016
 * 
 * $Revision$
 * 
 * $Date$
 * 
 * Copyright:
 * Werum Software & Systems AG
 * Wulf-Werum-Strasse 3, 
 * 21337 Lueneburg
 */
#ifndef _UV_KP_H_
#define _UV_KP_H_

#ifndef UVKP_API
#define UVKP_API __declspec(dllexport)
#endif

/**
 * \mainpage UV-KP C++-Bibliothek
 * Diese Bibliothek bietet Funktionen fr die Ausfhrung der UV-KP spezifischen Plausibilittsprfungen.
 * <P>
 * Beispiel fr die Ausfhrung der DSLN-Prfung:
 * \code
 *
 * #include <iostream>
 * #include <UV_KP.h>
 * 
 * using namespace std;
 * using namespace uvkp;
 * 
 * int main(int argc, char* argv[])
 * {
 *    // Pruefung ausfhren
 *    KernpruefungDSLN kernpruefung; // DSLN-Prfung
 *    Rueckgabe* rueckgabe = kernpruefung.pruefe(argv[1]);
 *
 *    // Alle Meldungen einzeln ausgeben
 *    cout << rueckgabe->getAnzahlRueckgabeMeldungen() << " Meldungen:" << endl;
 *    for (unsigned int i = 0; i < rueckgabe->getAnzahlRueckgabeMeldungen(); i++)
 *    {
 *       Meldung* meldung = rueckgabe->getRueckgabeMeldung(i);
 *       cout << meldung->toString() << endl;
 *    }
 *    
 *    // Alle Meldungen als DBFE-Bausteine ausgeben
 *    char* rueckgabeString = rueckgabe->toString();
 *    cout << rueckgabeString << endl;
 *    delete[] rueckgabeString; // rueckgabeString lschen und den Speicher freigeben
 *
 *    delete rueckgabe; // rueckgabe lschen und den Speicher freigeben
 * }
 *
 * \endcode
 */

/**
 * Beinhaltet C++-Klassen und -Funktionen fr die Durchfhrung von Plausibilittsprfungen
 * im Rahmen des Prfverfahrens UV-KP (Kernprfungen).
 */
namespace uvkp
{
   /**
    * Schweregrad einer Meldung.
    */
   enum Schweregrad
   {
      /**
       * Es handelt sich um einen Hinweis.
       */
      HINWEIS = 1,

      /**
       * Es handelt sich um einen Fehler.
       */
      FEHLER = 2,

      /**
       * Es handelt sich um einen Abbruch.
       */
      ABBRUCH = 4
   };

   /**
    * Meldung mit {@link Schweregrad} und Beschreibung.
    */
   class UVKP_API Meldung
   {
   public:
      /**
       * Gibt den Schweregrad der Meldung zurck.
       * @return Schweregrad der Meldung.
       */
      virtual Schweregrad getSchweregrad() = 0;

      /**
       * Gibt den Fehlerkode zurck.
       * @return Fehlerkode.
       */
      virtual char* getFehlerkode() = 0;

      /**
       * Gibt den Fehlertext der Meldung zurck.
       * @return Fehlertext.
       */
      virtual char* getFehlertext() = 0;

      /**
       * Gibt den Feldnamen zurck.
       * @return Feldname.
       */
      virtual char* getFeld() = 0;

      /**
       * Gibt die gesamte Meldung als String zurck.
       * @return Gesamte Meldung als String.
       */
      char* toString();
   };

   /**
    * Das Interface Rueckgabe enthlt alle Methoden, die fr die Abfrage des
    * Ergebnisses der Kernprfung notwendig sind.
    */
   class UVKP_API Rueckgabe
   {
   public:
     /**
      * Gibt den Return-Code der Kernprfung zurck.
      * <p>
      * Es werden folgende Return-Code festgelegt:
      * <ul>
      * 	<li>0 = Kernprfung fehlerfrei Ausgabe von getAnzahlRueckgabeMeldungen() ist 0
      * 	<li>1 = Kernprfung enthlt Hinweise
      * 	<li>2 = Kernprfung enthlt Fehler
      * 	<li>3 = Kernprfung enthlt Fehler und Hinweise
      * 	<li>4 = Kernprfung ist abgebrochen
      * </ul>
      * @return Return-Code
      */
      virtual unsigned int getReturnCode() = 0;

      /**
       * Gibt den vierstelliegen Typ der verwendeten Kernprfung zurck.
       * @return Vierstellieger Typ der verwendeten Kernprfung.
       */
      virtual char* getType() = 0;

      /**
       * Gibt die Version der verwendeten Kernprfung zurck.
       * @return Version der verwendeten Kernprfung.
       */
      virtual char* getVersion() = 0;

     /**
      * Gibt die Anzahl der Rckgabemeldungen zurck.
      * <p>
      * Es sind maximal neun Eintrge mglich.
      * @return Anzahl der Rckgabemeldungen
      */
     virtual unsigned int getAnzahlRueckgabeMeldungen() = 0;

     /**
      * Gibt die Rckgabemeldung mit dem vorgegebenem Index zurck.
      * @return {@link Meldung} mit dem vorgegebenem Index.
      */
     virtual Meldung* getRueckgabeMeldung(unsigned int i) = 0;

     /**
      * Gibt die String-Darstellung eines Rckgabeobjektes zurck.
      * <p>
      * <table border="1" frame="box" cellspacing="0">
      *   <caption>Aufbau des String's</caption>
      *   <tr bgcolor="#A8B8D9"><th colspan="2">Stelle</th><th rowspan="2">Inhalt</th></tr>
      *   <tr bgcolor="#A8B8D9"><th width="60">von</th><th width="60">bis</th></tr>
      *   <tr><td align="right">  1</td><td align="right">  1</td><td>Return-Code</td></tr>
      *   <tr><td align="right">  2</td><td align="right">  2</td><td>Anzahl der DBFE-Bausteine (max. 9)</td></tr>
      *   <tr><td align="right">  3</td><td align="right"> 75</td><td>DBFE-Baustein 1 (falls vorhanden)</td></tr>
      *   <tr><td align="right"> 76</td><td align="right">148</td><td>DBFE-Baustein 2 (falls vorhanden)</td></tr>
      *   <tr><td align="right">149</td><td align="right">221</td><td>DBFE-Baustein 3 (falls vorhanden)</td></tr>
      *   <tr><td align="right">222</td><td align="right">294</td><td>DBFE-Baustein 4 (falls vorhanden)</td></tr>
      *   <tr><td align="right">295</td><td align="right">367</td><td>DBFE-Baustein 5 (falls vorhanden)</td></tr>
      *   <tr><td align="right">368</td><td align="right">440</td><td>DBFE-Baustein 6 (falls vorhanden)</td></tr>
      *   <tr><td align="right">441</td><td align="right">513</td><td>DBFE-Baustein 7 (falls vorhanden)</td></tr>
      *   <tr><td align="right">514</td><td align="right">586</td><td>DBFE-Baustein 8 (falls vorhanden)</td></tr>
      *   <tr><td align="right">587</td><td align="right">659</td><td>DBFE-Baustein 9 (falls vorhanden)</td></tr>
      * </table>
      * @return String-Darstellung des Rckgabeobjektes
      */
      char* toString();
   };

   /**
    * Das Interface Kernpruefung enthlt die Methode zur Kernprfung eines Datensatzes.
    */
   class UVKP_API Kernpruefung
   {
   public:
      /**
       * Gibt den vierstelliegen Typ der Kernprfung zurck.
       * @return Vierstellieger Typ der Kernprfung.
       */
      virtual char* getType() = 0;

      /**
       * Der eingegebene Datensatz wird kerngeprft.
       * @param datensatz Datensatz als Zeichenkette.
       * @return {@link Rueckgabe}-Objekt.
       */
      Rueckgabe* pruefe(const char* datensatz);

   private:
      virtual char* getTypeErrorCode() = 0;
      virtual char* getTypeErrorText() = 0;
      virtual char* getFormatErrorCode() = 0;
      virtual char* getFormatErrorText() = 0;
      virtual void checkVersion(const char* version) = 0;
      virtual void checkMMDBAP(const char* datensatz) = 0;
   };

   /**
    * Implementierung des Interfaces {@link Kernpruefung} fr die Prfung von DSLN-Datenstzen.
    */
   class UVKP_API KernpruefungDSLN : public Kernpruefung
   {
   public:
      char* getType();
   private:
      char* getTypeErrorCode();
      char* getTypeErrorText();
      char* getFormatErrorCode();
      char* getFormatErrorText();
      void checkVersion(const char* version);
      void checkMMDBAP(const char* datensatz);
   };

   /**
    * Implementierung des Interfaces {@link Kernpruefung} fr die Prfung von DSAS-Datenstzen.
    */
   class UVKP_API KernpruefungDSAS : public Kernpruefung
   {
   public:
      char* getType();
   private:
      char* getTypeErrorCode();
      char* getTypeErrorText();
      char* getFormatErrorCode();
      char* getFormatErrorText();
      void checkVersion(const char* version);
      void checkMMDBAP(const char* datensatz);
   };

   /**
    * Implementierung des Interfaces {@link Kernpruefung} fr die Prfung von DSKO-Datenstzen.
    */
   class UVKP_API KernpruefungDSKO : public Kernpruefung
   {
   public:
      char* getType();
   private:
      char* getTypeErrorCode();
      char* getTypeErrorText();
      char* getFormatErrorCode();
      char* getFormatErrorText();
      void checkVersion(const char* version);
      void checkMMDBAP(const char* datensatz);
   };
}

#endif